/**************************************************************************************

Copyright (c) Hilscher Gesellschaft fuer Systemautomation mbH. All Rights Reserved.

***************************************************************************************

  $Id:  $:

  Description:
    Definition file for the CifX Device base class,

  Changes:
    Date        Description
    -----------------------------------------------------------------------------------
    2006-06-28  initial version

**************************************************************************************/
#pragma once

///////////////////////////////////////////////////////////////////////////////////////////
/// \file CifxDeviceBase.h
///  Definition file for the CifX Device base class,
///////////////////////////////////////////////////////////////////////////////////////////

#include "Cifxuser.h"
#include "CifxErrors.h"
#include <vector>
#include "cifXAPIlib.h"
#include <stdint.h>

#define CIFX_INFO_CHANNEL   0
#define CIFX_INFO_CHANNELIO 1

///////////////////////////////////////////////////////////////////////////////////////////
/// CifX Device base class
///////////////////////////////////////////////////////////////////////////////////////////
class CCifXDeviceBase
{
public:

  /////////////////////////////////////////////////////
  /// Structure for displaying the different properties of a device
  /////////////////////////////////////////////////////
  typedef struct PROPERTY_LISTtag
  {
    CString csProperty;     //!< name of the property
    CString csValue;        //!< value of the property

  } PROPERTY_LIST;

  typedef std::vector<CCifXDeviceBase::PROPERTY_LIST> PROPERTY_VECTOR;

  CCifXDeviceBase(BOARD_INFORMATION* ptBoardInfo, CHANNEL_INFORMATION* ptChannelInfo, CCifXAPIlib* pcCifXAPI);
  virtual ~CCifXDeviceBase(void);

  /////////////////////////////////////////////////////
  // Abstract functions
  /////////////////////////////////////////////////////
  virtual bool            IsSystemDevice(void)  = 0;
  virtual int32_t            OpenDevice(void)      = 0;
  virtual int32_t            CloseDevice(void)     = 0;
  virtual int32_t            PutPacket(CIFX_PACKET* ptPacket, uint32_t ulTimeout) = 0;
  virtual int32_t            GetPacket(CIFX_PACKET* ptPacket, uint32_t ulBufferSize, uint32_t ulTimeou) = 0;
  virtual uint32_t   GetChannelNr(void)    = 0;
  virtual PROPERTY_VECTOR GetChannelInformation(uint32_t ulCmd) = 0;
  virtual CString         GetChannelInfoName(void) = 0;
  virtual int32_t            Download(uint32_t ulChannel, uint32_t ulMode, char* szFileName, unsigned char* pabFileData, uint32_t ulFileSize, PFN_PROGRESS_CALLBACK pfnCallback, void* pvUser) = 0;
  virtual int32_t            Upload(char* szFilename, uint32_t ulChannel, unsigned char* pbBuffer, uint32_t* pulFileSize) = 0;

  virtual int32_t            GetMBXState(uint32_t* pulRecvCount, uint32_t* pulSendCount) = 0;
  virtual int32_t            Reset(uint32_t ulMode, uint32_t ulTimeout, uint32_t ulResetParam) = 0;
  /////////////////////////////////////////////////////

  virtual int32_t          IOInfo(uint32_t /*ulAreaNumber*/,
                               uint32_t /*ulOffset*/,
                               uint32_t /*ulDataLen*/,
                               void* /*pvData*/)             { return CIFX_FUNCTION_FAILED; }


  virtual int32_t          IORead(uint32_t /*ulAreaNumber*/,
                               uint32_t /*ulOffset*/,
                               uint32_t /*ulDataLen*/,
                               void* /*pvData*/,
                               uint32_t /*ulTimeout*/)  { return CIFX_FUNCTION_FAILED; }

  virtual int32_t          IOWrite(uint32_t /*ulAreaNumber*/,
                               uint32_t /*ulOffset*/,
                               uint32_t /*ulDataLen*/,
                               void* /*pvData*/,
                               uint32_t /*ulTimeout*/) { return CIFX_FUNCTION_FAILED; }
  virtual int32_t          IOReadSend(uint32_t /*ulAreaNumber*/,
                               uint32_t /*ulOffset*/,
                               uint32_t /*ulDataLen*/,
                               void* /*pvData*/)            { return CIFX_FUNCTION_FAILED; }

  virtual int32_t          Watchdog(uint32_t /*ulCmd*/, uint32_t* /*pulTrigger*/)  { return CIFX_FUNCTION_FAILED; }
  virtual int32_t          HostState(uint32_t /*ulCmd*/, uint32_t* /*pulState*/, uint32_t /*ulTimeout*/)  { return CIFX_FUNCTION_FAILED; }
  virtual int32_t          BusState(uint32_t /*ulCmd*/, uint32_t* /*pulState*/, uint32_t /*ulTimeout*/)  { return CIFX_FUNCTION_FAILED; }
  virtual int32_t          DMAState(uint32_t /*ulCmd*/, uint32_t* /*pulState*/)  { return CIFX_FUNCTION_FAILED; }
  virtual int32_t          ConfigLock(uint32_t /*ulCmd*/, uint32_t* /*pulState*/, uint32_t /*ulTimeout*/)  { return CIFX_FUNCTION_FAILED; }

  virtual int32_t          FindFirstFile(uint32_t ulChannel, CIFX_DIRECTORYENTRY* ptDirectoryInfo) = 0;
  virtual int32_t          FindNextFile(uint32_t ulChannel, CIFX_DIRECTORYENTRY* ptDirectoryInfo)  = 0;

  virtual std::vector<PROPERTY_LIST> GetDescription(void);

  /////////////////////////////////////////////////////////////////////////////
  /// Gets the drivers error description of the passed error
  ///   \param lError Error to look up
  ///   \return Error in human readable text
  /////////////////////////////////////////////////////////////////////////////
  CString GetErrorDescription(int32_t lError)
  {
    CString csRet;
    UNREFERENCED_PARAMETER(lError);
    m_pcCifXAPI->xDriverGetErrorDescription(lError, csRet.GetBuffer(MAX_PATH), MAX_PATH);
    csRet.ReleaseBuffer();

    return csRet;
  }

  /////////////////////////////////////////////////////////////////////////////
  /// Gets the full device name for displaying in a tree/dialog (<Name> (<alias>))
  ///   \return String containing device name
  /////////////////////////////////////////////////////////////////////////////
  virtual CString GetDeviceName(void)
  {
    USES_CONVERSION;
    CString csRet;

    if(strlen(m_tBoardInfo.abBoardAlias) > 0)
      csRet.Format(_T("%s (%s)"), A2T(m_tBoardInfo.abBoardName), A2T(m_tBoardInfo.abBoardAlias));
    else
      csRet.Format(_T("%s"), A2T(m_tBoardInfo.abBoardName));

    return csRet;
  }

  /////////////////////////////////////////////////////////////////////////////
  /// Gets the short device name for displaying in a tree/dialog (<Name> (<alias>))
  ///   \return String containing short device name
  /////////////////////////////////////////////////////////////////////////////
  virtual CString GetShortDeviceName(void)
  {
    return GetDeviceName();
  }

  /////////////////////////////////////////////////////////////////////////////
  /// Query the mailbox size of a device
  ///   \return Mailbox size of the device
  /////////////////////////////////////////////////////////////////////////////
  virtual uint32_t GetMailboxSize(void)  { return m_tChannelInfo.ulMailboxSize; }



  BOARD_INFORMATION*    GetBoardInfo(void)    { return &m_tBoardInfo;   }
  CHANNEL_INFORMATION*  GetChannelInfo(void)  { return &m_tChannelInfo; }

protected:
  BOARD_INFORMATION       m_tBoardInfo;     //!< Board information returned from cifX driver
  CHANNEL_INFORMATION     m_tChannelInfo;   //!< Channel information returned from cifX driver (if available)
  CCifXAPIlib*            m_pcCifXAPI;
};
